//----------------------------------------------------------------------------
//
// Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
//
// Use, modification and distribution are subject to the Boost Software
// License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
//
//----------------------------------------------------------------------------

// This is an example program for using the COM interface of SIMCA-Q dll. To build and 
// run this application you must copy SIMCAQ.tlb to the same directory where you have 
// put the source files.
//
// You must also register the SIMCA-Q.dll with regsvr32.

#include "stdafx.h"
#include "SQMCOMSample.h"
#include "SQMCOMSampleDlg.h"

#include <fstream>
#include <string>
#include <vector>

/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
   CAboutDlg() : CDialog(CAboutDlg::IDD) {}

   enum { IDD = IDD_ABOUTBOX };
};

//////////////////////////////////////////////////////////////////////////////
// CSQMCOMSampleDlg dialog

CSQMCOMSampleDlg::CSQMCOMSampleDlg(CWnd* pParent /*=NULL*/)
   : CDialog(CSQMCOMSampleDlg::IDD, pParent)
   , m_hIcon(AfxGetApp()->LoadIcon(IDR_MAINFRAME))
{
}

void CSQMCOMSampleDlg::DoDataExchange(CDataExchange* pDX)
{
   CDialog::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(CSQMCOMSampleDlg, CDialog)
   ON_WM_SYSCOMMAND()
   ON_WM_PAINT()
   ON_WM_QUERYDRAGICON()
   ON_BN_CLICKED(IDC_BUTTON1, OnBnClickedButton1)
END_MESSAGE_MAP()

BOOL CSQMCOMSampleDlg::OnInitDialog()
{
   CDialog::OnInitDialog();

   // Add "About..." menu item to system menu.

   // IDM_ABOUTBOX must be in the system command range.
   ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
   ASSERT(IDM_ABOUTBOX < 0xF000);

   CMenu* pSysMenu = GetSystemMenu(FALSE);
   if (pSysMenu != nullptr)
   {
      CString strAboutMenu;
      strAboutMenu.LoadString(IDS_ABOUTBOX);
      if (!strAboutMenu.IsEmpty())
      {
         pSysMenu->AppendMenu(MF_SEPARATOR);
         pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
      }
   }

   // Set the icon for this dialog.  The framework does this automatically
   //  when the application's main window is not a dialog
   SetIcon(m_hIcon, TRUE);			// Set big icon
   SetIcon(m_hIcon, FALSE);		// Set small icon

   // TODO: Add extra initialization here

   return TRUE;  // return TRUE  unless you set the focus to a control
}

void CSQMCOMSampleDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
   if ((nID & 0xFFF0) == IDM_ABOUTBOX)
   {
      CAboutDlg dlgAbout;
      dlgAbout.DoModal();
   }
   else
   {
      CDialog::OnSysCommand(nID, lParam);
   }
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CSQMCOMSampleDlg::OnPaint()
{
   if (IsIconic())
   {
      CPaintDC dc(this); // device context for painting

      SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

      // Center icon in client rectangle
      int cxIcon = GetSystemMetrics(SM_CXICON);
      int cyIcon = GetSystemMetrics(SM_CYICON);
      CRect rect;
      GetClientRect(&rect);
      int x = (rect.Width() - cxIcon + 1) / 2;
      int y = (rect.Height() - cyIcon + 1) / 2;

      // Draw the icon
      dc.DrawIcon(x, y, m_hIcon);
   }
   else
   {
      CDialog::OnPaint();
   }
}

// The system calls this function to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CSQMCOMSampleDlg::OnQueryDragIcon()
{
   return static_cast<HCURSOR>(m_hIcon);
}

void CSQMCOMSampleDlg::OnBnClickedButton1()
{
   CString strErr;
   CString strUspPath;              // Will hold the full path to the .usp file to add.
   CString strOutputFile;           // Will hold the full path to the log file.

   wchar_t szTempPath[_MAX_DIR];
   GetTempPathW(_MAX_DIR, szTempPath);
   strOutputFile = szTempPath + CString(L"SIMCA-Q COM result.txt");

   try
   {
      CLSID clsSQ = {0};
      if ((CLSIDFromProgID(L"Umetrics.SIMCAQ", &clsSQ)) != S_OK)
         return;

      const auto hr = mpSQ.CreateInstance(clsSQ);
      if (hr != S_OK)
         throw("CreateInstance failed");

      // mpFile is a temp file where we store the results.
      mpFile = _wfopen(strOutputFile.GetString(), L"w, ccs=UTF-8");

      if (mpFile == nullptr)
      {
         throw "Could not open the result file.";
      }

      // Let the user specify a text file to create a SIMCA project from.
      CFileDialog oOpenDlg(true, nullptr, nullptr,
         OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_FILEMUSTEXIST | OFN_ENABLESIZING,
         L"Text file (*.txt)|*.txt|All files (*.*)|*.*||");

      oOpenDlg.m_ofn.lpstrTitle = L"Import Text File";
      if (IDOK == oOpenDlg.DoModal())
      {
         // Now an *.txt file has been selected.
         CString strTxtPath = oOpenDlg.GetPathName();

         fwprintf(mpFile, L"Path to txt file:        %s\n", strTxtPath.GetString());

         // Let the user select a name and path where to store the SIMCA project.
         CFileDialog oSaveDlg(false, L".usp",
            oOpenDlg.GetFileTitle(),
            OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_ENABLESIZING,
            L"SIMCA project (*.usp)|*.usp|All files (*.*)|*.*||");

         oSaveDlg.m_ofn.lpstrTitle = _T("Save SIMCA Project");

         if (IDOK == oSaveDlg.DoModal())
         {
            CWaitCursor oWait;

            // Now a name for the *.usp file has been selected.
            strUspPath = oSaveDlg.GetPathName();

            // If the file already exist we need to remove it or creating the SIMCA project will fail.
            if (_waccess(strUspPath.GetString(), 0) == 0)
            {
               if (_wunlink(strUspPath.GetString()) == -1)
               {
                  throw "The SIMCA project file is already open in another program.";
               }
            }

            fwprintf(mpFile, L"Path to usp file:        %s\n", strUspPath.GetString());

            // Import the file as a SIMCA project.
            Import(strTxtPath, strUspPath);

            if (mpProject)
            {
               CreateWorkset();

               FitModel();

               mpProject->Save();
            }
         }
      }
   }
   catch (const char* szError)
   {
      strErr = szError;
   }
   catch (const _com_error& err)
   {
      strErr.Append(mpSQ->GetErrorDescription(err.Error()));
   }
   catch (...)
   {
      strErr = "Unknown error";
   }

   if (mpFile)
   {
      fclose(mpFile);
      mpFile = nullptr;
   }

   if (strErr.GetLength() > 0)
   {
      AfxMessageBox(strErr);
   }
   else if (!strUspPath.IsEmpty())
   {
      if (auto pidl = ILCreateFromPath(strUspPath.GetString()))
      {
         SHOpenFolderAndSelectItems(pidl, 0, nullptr, 0);
         ILFree(pidl);
      }

      if (_waccess(strOutputFile.GetString(), 00) == 0)
      {
         ::ShellExecuteW(nullptr, L"open", strOutputFile.GetString(), nullptr, nullptr, SW_SHOWNORMAL);
      }
   }

   if (mpProject)
      mpProject->DisposeProject();
}

std::vector<std::string> Split(const std::string& rstrJoined, const std::string& rstrSeparator)
{
   std::vector<std::string> strvecSplitted;
   std::string::size_type uPrevPos = 0;
   for (std::string::size_type uPos = rstrJoined.find(rstrSeparator); uPos != std::string::npos; uPos = rstrJoined.find(rstrSeparator, uPos))
   {
      strvecSplitted.push_back(rstrJoined.substr(uPrevPos, uPos - uPrevPos));
      uPos += rstrSeparator.length();
      uPrevPos = uPos;
   }

   strvecSplitted.push_back(rstrJoined.substr(uPrevPos, rstrJoined.length() - uPrevPos));
   return strvecSplitted;
}

std::wstring utf8_decode(const std::string& strFrom)
{
   if (strFrom.empty())
      return {};
   int nDestLen = MultiByteToWideChar(CP_UTF8, 0, &strFrom[0], static_cast<int>(strFrom.size()), nullptr, 0);
   std::wstring wstrTo(nDestLen, 0);
   MultiByteToWideChar(CP_UTF8, 0, &strFrom[0], static_cast<int>(strFrom.size()), &wstrTo[0], nDestLen);
   return wstrTo;
}


void CSQMCOMSampleDlg::Import(const wchar_t* szTxtFile, const wchar_t* szUspFile)
{
   // Imports a text file. The code assumes that the file is in UTF-8 and of the format given 
   // by the constants iNumObsIDs, iNumVarIDs and iNumVarIDs below.

   HRESULT hr = S_FALSE;
   const long iNumObsIDs = 2;  // The import demands that the input file contains two observation ids.
   const long iNumVarIDs = 2;  // The import demands that the input file contains two variable ids.
   const long iNumQualVar = 2; // The import demands that the input file contains two qualitative variables.
   long iNumRows = 0;
   long iNumCols = 0;
   float    fVal;
   _bstr_t  empty;

   IImportPtr       pImport;
   IStringMatrixPtr pstrMatObservationIDs;
   IFloatMatrixPtr  pfMatData;
   IStringMatrixPtr pstrMatQualData;
   IStringMatrixPtr pstrMatVariableIDs;
   IStringMatrixPtr pstrMatQualVariableIDs;
   IStringVectorPtr pstrVecNames;
   IStringVectorPtr pstrVecObsNames;

   std::ifstream oIn(szTxtFile, std::ios::in | std::ios::binary);

   if (!oIn)
      throw "The input file could not be opened.";

   std::string strLine;
   std::vector<std::vector<std::string>> strvecMatrix;
   while (std::getline(oIn, strLine))
   {
      strLine.erase(strLine.find_last_not_of(" \n\r\t") + 1);
      strvecMatrix.push_back(Split(strLine, "\t"));
   }

   if (strvecMatrix.empty())
      throw "The input file is empty.";

   iNumRows = static_cast<long>(strvecMatrix.size());
   iNumCols = static_cast<long>(strvecMatrix[0].size());

   if (iNumRows - iNumVarIDs < 3)
      throw "There must be at least three rows of data in the input matrix.";

   if (iNumCols - iNumObsIDs < 3)
      throw "There must be at least three columns of data in the input matrix.";

   for (const auto& row : strvecMatrix)
   {
      if (static_cast<long>(row.size()) != iNumCols)
         throw "The rows in the input file have uneven number of columns.";
   }

   pImport = mpSQ->InitImport(_bstr_t(szUspFile), empty);

   /* Initiate a matrix for the observation ids,
   Number of rows = total number of rows - number of variable ids
   Number of columns = number of observation ids */
   pstrMatObservationIDs = mpSQ->GetNewStringMatrix(iNumRows - iNumVarIDs, iNumObsIDs);

   /* Initiate a float matrix for the quantitative data,
   Number of rows = total number of rows - number of variable ids
   Number of columns = total number of columns - (number of qualitative variables + number of observation ids)*/
   pfMatData = mpSQ->GetNewFloatMatrix(iNumRows - iNumVarIDs, iNumCols - (iNumQualVar + iNumObsIDs));

   /* Initiate a string matrix for the qualitative data,
   Number of rows = total number of rows - number of variable ids
   Number of columns = number of qualitative variables */
   if (iNumQualVar > 0)
   {
      pstrMatQualData = mpSQ->GetNewStringMatrix(iNumRows - iNumVarIDs, iNumQualVar);
   }

   /* Initiate a string matrix for the variable ids,
   Number of rows = number of variable ids
   Number of columns = Total number of columns - (number of observation ids + number of qualitative variables)*/
   pstrMatVariableIDs = mpSQ->GetNewStringMatrix(iNumVarIDs, iNumCols - (iNumQualVar + iNumObsIDs));

   /* Initiate a string matrix for the qualitative variable ids,
   Number of rows = number of variable ids
   Number of columns = number of qualitative variables */
   if (iNumQualVar > 0)
   {
      pstrMatQualVariableIDs = mpSQ->GetNewStringMatrix(iNumVarIDs, iNumQualVar);
   }

   /* Initiate a string vector for the variable id names,
   Number of strings = number of variable ids */
   pstrVecNames = mpSQ->GetNewStringVector(iNumVarIDs);

   /* Initiate a string vector for the observation id names,
   Number of strings = Number of observation ids */
   pstrVecObsNames = mpSQ->GetNewStringVector(iNumObsIDs);

   for (long iRow = 1; iRow <= iNumRows; ++iRow)
   {
      for (long iCol = 1; iCol <= iNumCols; ++iCol)
      {
         _bstr_t  bstr = utf8_decode(strvecMatrix[iRow - 1][iCol - 1]).c_str();

         if (iRow == 1 && iCol <= iNumObsIDs)
         {
            /* Set the names of the observation ids. */
            hr = pstrVecObsNames->SetData(iCol, bstr);
            if (iCol == 1 && SUCCEEDED(hr))
            {
               /* Set the names of the variable ids. */
               hr = pstrVecNames->SetData(iRow, bstr);
            }
         }
         else if (iCol == 1 && iRow <= iNumVarIDs)
         {
            /* Set the names of the variable ids. */
            hr = pstrVecNames->SetData(iRow, bstr);
         }
         else if (iCol > iNumObsIDs && iCol <= (iNumObsIDs + iNumQualVar) && iRow <= iNumVarIDs)
         {
            /* Set the names of the qualitative variables. */
            hr = pstrMatQualVariableIDs->SetData(iRow, iCol - iNumObsIDs, bstr);
         }
         else if (iCol > (iNumObsIDs + iNumQualVar) && iRow <= iNumVarIDs)
         {
            /* Set the names of the quantitative variables. */
            hr = pstrMatVariableIDs->SetData(iRow, iCol - (iNumObsIDs + iNumQualVar), bstr);
         }
         else if (iCol <= iNumObsIDs && iRow > iNumVarIDs)
         {
            /* Set the names of the observations. */
            hr = pstrMatObservationIDs->SetData(iRow - iNumVarIDs, iCol, bstr);
         }
         else if (iCol <= (iNumObsIDs + iNumQualVar) && iRow > iNumVarIDs)
         {
            /* Set the qualitative data. */
            hr = pstrMatQualData->SetData(iRow - iNumVarIDs, iCol - iNumObsIDs, bstr);
         }
         else if (iRow > iNumVarIDs && iCol > (iNumObsIDs + iNumQualVar))
         {
            /* Set quantitative data */
            fVal = static_cast<float>(atof(static_cast<const char*>(bstr)));
            hr = pfMatData->SetData(iRow - iNumVarIDs, iCol - (iNumObsIDs + iNumQualVar), fVal);
         }
         if (!SUCCEEDED(hr))
            throw "Import failed when reading input file.";
      }
   }

   /*Reserve space for the import, this will make the import faster if you call*/
   pImport->Reserve(iNumRows - iNumVarIDs, iNumCols - (iNumObsIDs + iNumQualVar), iNumQualVar);

   /*Set the observation names in the new dataset.*/
   pImport->SetObservationNames(pstrMatObservationIDs, 1);

   /*Set a value the the import will treat as a missing value.*/
   pImport->SetMissingValueRepresentation(-99);

   /*Set a value the the import will treat as a missing value.*/
   pImport->SetMissingValueStringRepresentation("N/A");

   /*Add variables to the new dataset.*/
   pImport->AddQuantitativeVariables(pfMatData, pstrMatVariableIDs, 1);

   /*Add qualitative vars to variables to the new dataset.*/
   if (iNumQualVar > 0)
   {
      pImport->AddQualitativeVariables(pstrMatQualData, pstrMatQualVariableIDs, 1);
   }

   /*Name the secondary IDs, the primary ID is always called PrimaryID an can't be renamed*/
   pImport->SetVariableIDSeriesNames(pstrVecNames);

   /*Name the secondary IDs, the primary ID is always called PrimaryID an can't be renamed*/
   pImport->SetObservationIDSeriesNames(pstrVecObsNames);

   /*Set the name of the new dataset*/
   pImport->SetDataSetName(_bstr_t("Primary dataset"));

   /*Create a dataset of the variables and observations added.*/
   mpProject = pImport->FinishImport();
}

void CSQMCOMSampleDlg::CreateWorkset()
{
   IIntVectorPtr pDatasets = mpSQ->GetNewIntVector(1);
   pDatasets->SetData(1, 1);
   IWorksetPtr pWorkset;
   pWorkset = mpProject->GetNewWorkset(pDatasets);
   pWorkset->CreateDefaultWorkset();
   SaveModel(pWorkset);
}

void CSQMCOMSampleDlg::SaveModel(IWorksetPtr pWorkset)
{
   pWorkset->SetWorksetDescription("Created by SQMCOMSample");

   /* Create a new model for the workset. */
   IIntVectorPtr piModels = pWorkset->CreateModel(FALSE, ePLS);
   IModelPtr pModel = mpProject->GetModel(piModels->GetData(1));

   fwprintf(mpFile, L"\nModel number %i created.\n", pModel->GetModelNumber());
}

void CSQMCOMSampleDlg::FitModel()
{
   /* Autofit the first model.  */

   IModelPtr pModel = mpProject->GetModel(1);
   pModel->AutofitModel();
   fwprintf(mpFile, L"AutofitModel number 1 OK.\n");

   int iNumComponents = pModel->GetNumberOfComponents();
   /* Make sure that the model has at least two components after this sample has been run. */
   for (int iComp = iNumComponents; iComp < 2; ++iComp)
   {
      pModel->CalculateNextComponent();
      fwprintf(mpFile, L"CalculateNextComponent model number 1 OK.\n");
   }
}
